# 🚀 VPN System - Professional Scalable Architecture

Complete VPN management system with PHP/MySQL backend, REST API, and Android VPN client using VPN Gate OpenVPN configs.

## 📋 Table of Contents

- [System Overview](#system-overview)
- [Features](#features)
- [Tech Stack](#tech-stack)
- [Installation](#installation)
- [Configuration](#configuration)
- [Usage Guide](#usage-guide)
- [API Documentation](#api-documentation)
- [Android App Setup](#android-app-setup)
- [Security](#security)
- [Troubleshooting](#troubleshooting)

## 🎯 System Overview

```
VPN Gate (.ovpn files)
        ↓
Admin Panel (PHP + MySQL)
        ↓
REST API (Multiple API Keys)
        ↓
Android VPN Apps (Java)
        ↓
OpenVPN Core (ics-openvpn)
```

## ✨ Features

### Admin Panel
- ✅ Secure login with bcrypt password hashing
- ✅ Upload and auto-parse `.ovpn` files
- ✅ Manage multiple VPN servers
- ✅ Generate unlimited API keys
- ✅ Assign servers to specific APIs
- ✅ Set usage limits (daily/unlimited)
- ✅ Real-time API usage statistics
- ✅ Activity logging
- ✅ Modern premium UI/UX

### REST API
- ✅ API key authentication
- ✅ Usage limit enforcement
- ✅ Server filtering by API access
- ✅ Request logging
- ✅ JSON response format
- ✅ CORS support

### Android App
- ✅ Dynamic server loading
- ✅ Premium UI with animations
- ✅ OpenVPN integration ready
- ✅ No hardcoded servers
- ✅ Device ID tracking

## 🛠️ Tech Stack

**Backend:**
- PHP 7.4+
- MySQL 5.7+ / MariaDB
- PDO for database access
- Retrofit-style REST API

**Android:**
- Java
- Android SDK 21+
- Retrofit 2.9
- Lottie Animations
- Glide Image Loading
- ics-openvpn (integration required)

## 📦 Installation

### 1. Database Setup

Import the database:
```bash
mysql -u root -p < database.sql
```

Or create database manually:
```sql
CREATE DATABASE vpn_system CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;
```

Then import:
```bash
mysql -u root -p vpn_system < database.sql
```

### 2. Configure Database Connection

Edit `config/database.php`:
```php
define('DB_HOST', 'localhost');
define('DB_NAME', 'vpn_system');
define('DB_USER', 'root');
define('DB_PASS', 'your_password');
```

### 3. Configure Site URL

Edit `config/config.php`:
```php
define('SITE_URL', 'http://yourdomain.com/freevpn');
```

### 4. Set Permissions

```bash
chmod 755 config/
chmod 644 config/*.php
chmod 755 logs/
chmod 755 uploads/
```

### 5. Access Admin Panel

Navigate to: `http://yourdomain.com/freevpn/admin`

**Default Login:**
- Email: `admin@vpnsystem.com`
- Password: `admin123`

⚠️ **IMPORTANT:** Change the default password immediately!

## ⚙️ Configuration

### Generate Secure Admin Password

Run this to generate a new password hash:
```php
<?php
echo password_hash('your_new_password', PASSWORD_BCRYPT, ['cost' => 12]);
?>
```

Update in database:
```sql
UPDATE admins SET password = 'NEW_HASH_HERE' WHERE email = 'admin@vpnsystem.com';
```

### Upload VPN Servers

1. Log into admin panel
2. Go to "VPN Servers"
3. Click "Upload Server Config"
4. Fill in:
   - Server Name (e.g., "USA Server 1")
   - Country (e.g., "United States")
   - Flag URL (optional)
   - Upload `.ovpn` file from VPN Gate

### Generate API Keys

1. Go to "API Keys"
2. Click "Generate New API Key"
3. Set:
   - App Name (e.g., "Free VPN App")
   - Daily Usage Limit (or leave empty for unlimited)
4. Click "🌐 Servers" to assign servers to this API

## 📖 Usage Guide

### Admin Panel Flow

1. **Login** → Dashboard shows statistics
2. **Upload Servers** → Add VPN Gate configs
3. **Generate API** → Create API key for app
4. **Assign Servers** → Select which servers API can access
5. **Monitor** → View logs and usage stats

### API Testing

Visit: `http://yourdomain.com/freevpn/api`

Test with curl:
```bash
# Get servers
curl -H "X-API-Key: YOUR_API_KEY" \
     http://yourdomain.com/freevpn/api/servers.php

# Connect to server
curl -X POST \
     -H "X-API-Key: YOUR_API_KEY" \
     -H "Content-Type: application/json" \
     -d '{"server_id":1,"device_id":"test-device"}' \
     http://yourdomain.com/freevpn/api/connect.php
```

## 🔌 API Documentation

### Base URL
```
http://yourdomain.com/freevpn/api/
```

### Authentication
Include API key in every request:
```
Header: X-API-Key: YOUR_API_KEY
```

### Endpoints

#### GET /servers.php
Get list of available servers

**Response:**
```json
{
  "success": true,
  "data": {
    "servers": [
      {
        "id": 1,
        "server_name": "USA Server 1",
        "country": "United States",
        "flag_url": "https://...",
        "status": "active"
      }
    ],
    "total": 1
  }
}
```

#### POST /connect.php
Get OpenVPN configuration

**Request:**
```json
{
  "server_id": 1,
  "device_id": "unique-device-id"
}
```

**Response:**
```json
{
  "success": true,
  "data": {
    "server": {
      "id": 1,
      "server_name": "USA Server 1",
      "country": "United States",
      "ovpn_config": "client\nremote ..."
    }
  }
}
```

## 📱 Android App Setup

### 1. Configure API

Edit `ApiConfig.java`:
```java
public static final String BASE_URL = "http://yourdomain.com/freevpn/api/";
public static final String API_KEY = "YOUR_API_KEY_FROM_ADMIN_PANEL";
```

### 2. Add ics-openvpn Library

Download from: https://github.com/schwabe/ics-openvpn

Add to `settings.gradle`:
```gradle
include ':app', ':ics-openvpn-main'
```

Add dependency in `app/build.gradle`:
```gradle
implementation project(':ics-openvpn-main')
```

### 3. Build & Test

```bash
./gradlew assembleDebug
```

Install on device:
```bash
adb install app/build/outputs/apk/debug/app-debug.apk
```

## 🔒 Security

See [SECURITY.md](SECURITY.md) for detailed security practices.

**Quick Checklist:**
- ✅ Change default admin password
- ✅ Use HTTPS in production
- ✅ Set proper file permissions
- ✅ Enable error logging (disable display_errors)
- ✅ Use strong API keys
- ✅ Implement rate limiting
- ✅ Regular security updates

## 🐛 Troubleshooting

### Database Connection Failed
- Check credentials in `config/database.php`
- Verify MySQL service is running
- Check database exists

### Can't Upload .ovpn Files
- Check `upload_max_filesize` in php.ini
- Verify `uploads/` directory permissions
- Check file extension is exactly `.ovpn`

### API Returns 401
- Verify API key is correct
- Check API status is "active"
- Ensure servers are assigned to API

### Android App Can't Connect
- Verify BASE_URL is correct
- Check API_KEY matches admin panel
- Ensure `android:usesCleartextTraffic="true"` for HTTP
- Check device has internet connection

### VPN Won't Connect
- Verify ics-openvpn library is integrated
- Check .ovpn config is valid
- Ensure VPN permission is granted
- Test config manually with OpenVPN client

## 📂 Folder Structure

```
freevpn/
├── config/
│   ├── database.php
│   └── config.php
├── admin/
│   ├── login.php
│   ├── dashboard.php
│   ├── servers.php
│   ├── api-keys.php
│   ├── api-logs.php
│   ├── handlers/
│   ├── includes/
│   ├── css/
│   └── js/
├── api/
│   ├── servers.php
│   ├── connect.php
│   ├── helpers/
│   └── index.php
├── logs/
├── uploads/
├── database.sql
└── README.md

AndroidApp/
└── app/
    └── src/main/
        ├── AndroidManifest.xml
        ├── java/com/vpn/free/
        │   ├── MainActivity.java
        │   ├── SplashActivity.java
        │   ├── api/
        │   ├── models/
        │   ├── adapters/
        │   ├── openvpn/
        │   └── utils/
        └── res/
```

## 🚀 Scalability

System supports:
- ✅ Unlimited VPN servers
- ✅ Unlimited API keys (multiple apps)
- ✅ Millions of API requests
- ✅ Horizontal scaling (stateless API)
- ✅ Future iOS app support
- ✅ Payment integration ready

## 📄 License

This is a commercial project. All rights reserved.

## 🆘 Support

For issues and questions:
- Check documentation
- Review logs in `logs/activity.log`
- Test API with curl/Postman
- Verify database connections

---

**Built with ❤️ for scalable VPN systems**
