# 🔒 Security Best Practices

## Admin Panel Security

### 1. Change Default Password
**CRITICAL:** Change the default admin password immediately after installation.

Generate secure password hash:
```php
<?php
echo password_hash('YourStrong!Password123', PASSWORD_BCRYPT, ['cost' => 12]);
?>
```

Update database:
```sql
UPDATE admins SET password = 'NEW_HASH' WHERE email = 'admin@vpnsystem.com';
```

### 2. HTTPS Only
**Production Requirement:** Always use HTTPS.

Update `config/config.php`:
```php
define('SITE_URL', 'https://yourdomain.com/freevpn');
```

Force HTTPS in `.htaccess`:
```apache
RewriteEngine On
RewriteCond %{HTTPS} off
RewriteRule ^(.*)$ https://%{HTTP_HOST}%{REQUEST_URI} [L,R=301]
```

### 3. Session Security
Already implemented:
- ✅ HTTP-only cookies
- ✅ Session regeneration on login
- ✅ CSRF token protection

### 4. File Upload Security
Already implemented:
- ✅ Extension validation
- ✅ File size limits
- ✅ Content validation
- ✅ No direct execution

Additional recommendation:
```apache
# In uploads/.htaccess
Deny from all
```

### 5. Database Security
- ✅ Prepared statements (PDO)
- ✅ No dynamic SQL
- ✅ Input validation

**Recommendation:** Create dedicated MySQL user with limited privileges:
```sql
CREATE USER 'vpn_user'@'localhost' IDENTIFIED BY 'strong_password';
GRANT SELECT, INSERT, UPDATE, DELETE ON vpn_system.* TO 'vpn_user'@'localhost';
FLUSH PRIVILEGES;
```

### 6. Error Handling

**Production settings** in `config/config.php`:
```php
error_reporting(E_ALL);
ini_set('display_errors', 0);  // MUST be 0 in production
ini_set('log_errors', 1);
```

### 7. File Permissions
```bash
# Files
find . -type f -exec chmod 644 {} \;

# Directories
find . -type d -exec chmod 755 {} \;

# Sensitive configs
chmod 600 config/database.php

# Writable directories
chmod 755 logs/
chmod 755 uploads/
```

### 8. Admin Access Control

Restrict admin panel to specific IPs (optional):
```apache
# In admin/.htaccess
Order Deny,Allow
Deny from all
Allow from 1.2.3.4  # Your IP
Allow from 5.6.7.8  # Office IP
```

## API Security

### 1. API Key Generation
Already secure:
- ✅ 64-character random keys
- ✅ Cryptographically secure (`random_bytes()`)

### 2. Rate Limiting
Implemented:
- ✅ Per-API usage limits
- ✅ Daily reset mechanism
- ✅ Request logging

**Recommendation:** Add IP-based rate limiting:
```php
// In api/helpers/auth.php
// Check requests per IP in last minute
$ipRequests = $db->prepare("
    SELECT COUNT(*) FROM api_logs 
    WHERE ip_address = ? 
    AND request_time > DATE_SUB(NOW(), INTERVAL 1 MINUTE)
");
$ipRequests->execute([$_SERVER['REMOTE_ADDR']]);
if ($ipRequests->fetchColumn() > 60) {
    sendError('Rate limit exceeded', 'RATE_LIMIT', 429);
}
```

### 3. HTTPS for API
**Production Requirement:** API must use HTTPS.

Android app SSL pinning (recommended):
```java
// In ApiClient.java
CertificatePinner certificatePinner = new CertificatePinner.Builder()
    .add("yourdomain.com", "sha256/AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA=")
    .build();

OkHttpClient client = new OkHttpClient.Builder()
    .certificatePinner(certificatePinner)
    .build();
```

### 4. API Key Storage

**NEVER** commit API keys to version control.

Use environment variables:
```java
// BuildConfig approach
buildConfigField "String", "API_KEY", "\"${System.getenv('VPN_API_KEY')}\""
```

### 5. Request Validation
Already implemented:
- ✅ Input sanitization
- ✅ Type checking
- ✅ Required field validation

### 6. CORS Configuration

**Production:** Restrict origins in `api/*.php`:
```php
// Replace wildcard with specific domain
header('Access-Control-Allow-Origin: https://yourdomain.com');
```

## Android App Security

### 1. Secure Storage
Use Android Keystore for API key:
```java
// Use EncryptedSharedPreferences
SharedPreferences prefs = EncryptedSharedPreferences.create(
    context,
    "vpn_prefs",
    masterKey,
    EncryptedSharedPreferences.PrefKeyEncryptionScheme.AES256_SIV,
    EncryptedSharedPreferences.PrefValueEncryptionScheme.AES256_GCM
);
```

### 2. ProGuard/R8
Enable code obfuscation:
```gradle
buildTypes {
    release {
        minifyEnabled true
        shrinkResources true
        proguardFiles getDefaultProguardFile('proguard-android-optimize.txt')
    }
}
```

### 3. Network Security Config

Create `res/xml/network_security_config.xml`:
```xml
<?xml version="1.0" encoding="utf-8"?>
<network-security-config>
    <domain-config cleartextTrafficPermitted="false">
        <domain includeSubdomains="true">yourdomain.com</domain>
        <pin-set expiration="2025-12-31">
            <pin digest="SHA-256">AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA=</pin>
        </pin-set>
    </domain-config>
</network-security-config>
```

### 4. Input Validation
Always validate server responses:
```java
if (server.getOvpnConfig() != null && 
    server.getOvpnConfig().contains("client") &&
    server.getOvpnConfig().contains("remote")) {
    // Valid config
} else {
    // Invalid or malicious
}
```

### 5. Permissions
Use minimum required permissions:
- ✅ INTERNET
- ✅ ACCESS_NETWORK_STATE
- ✅ BIND_VPN_SERVICE

### 6. WebView Security
If using WebView:
```java
webView.getSettings().setJavaScriptEnabled(false);
webView.getSettings().setAllowFileAccess(false);
```

## Server Security

### 1. Keep Updated
```bash
# Regular updates
apt update && apt upgrade

# PHP updates
apt install php php-cli php-mysql php-curl
```

### 2. Firewall
```bash
# UFW example
ufw allow 80/tcp
ufw allow 443/tcp
ufw allow 22/tcp
ufw enable
```

### 3. Fail2Ban
```bash
apt install fail2ban

# Create /etc/fail2ban/jail.local
[DEFAULT]
bantime = 3600
findtime = 600
maxretry = 5
```

### 4. Database Security
```sql
-- Remove test database
DROP DATABASE IF EXISTS test;

-- Remove anonymous users
DELETE FROM mysql.user WHERE User='';

-- Disallow root login remotely
DELETE FROM mysql.user WHERE User='root' AND Host NOT IN ('localhost', '127.0.0.1', '::1');

FLUSH PRIVILEGES;
```

### 5. PHP Security

In `php.ini`:
```ini
expose_php = Off
display_errors = Off
log_errors = On
error_log = /var/log/php/error.log
disable_functions = exec,passthru,shell_exec,system,proc_open,popen
file_uploads = On
upload_max_filesize = 5M
post_max_size = 8M
max_execution_time = 30
max_input_time = 30
```

### 6. Disable Directory Listing

In `.htaccess`:
```apache
Options -Indexes
```

### 7. Protect Sensitive Files

```apache
<FilesMatch "\.(sql|log|md|gitignore|htaccess)$">
    Order allow,deny
    Deny from all
</FilesMatch>
```

## Monitoring & Logging

### 1. Activity Logging
Already implemented:
- ✅ Admin actions
- ✅ API requests
- ✅ Error logging

### 2. Log Rotation
```bash
# Create /etc/logrotate.d/vpnsystem
/var/www/html/freevpn/logs/*.log {
    daily
    rotate 14
    compress
    delaycompress
    missingok
    notifempty
}
```

### 3. Monitor Suspicious Activity
Check logs regularly:
```bash
# Failed login attempts
grep "Failed login" logs/activity.log

# API errors
grep "API key" logs/activity.log

# High usage
grep "LIMIT_EXCEEDED" logs/activity.log
```

## Incident Response

### 1. Compromised API Key
```sql
-- Disable immediately
UPDATE api_keys SET status = 'inactive' WHERE api_key = 'COMPROMISED_KEY';

-- Delete if necessary
DELETE FROM api_keys WHERE api_key = 'COMPROMISED_KEY';
```

### 2. Compromised Admin Account
```sql
-- Change password immediately
UPDATE admins SET password = 'NEW_HASH' WHERE email = 'admin@vpnsystem.com';

-- Review logs
SELECT * FROM api_logs WHERE DATE(request_time) = CURDATE();
```

### 3. Malicious Server Upload
```sql
-- Delete server
DELETE FROM vpn_servers WHERE id = 123;

-- Check affected APIs
SELECT * FROM api_server_access WHERE server_id = 123;
```

## Security Checklist

- [ ] Changed default admin password
- [ ] Enabled HTTPS
- [ ] Created dedicated database user
- [ ] Set proper file permissions
- [ ] Disabled PHP error display
- [ ] Enabled error logging
- [ ] Configured firewall
- [ ] Installed fail2ban
- [ ] Secured MySQL
- [ ] Restricted admin access (optional)
- [ ] Implemented SSL pinning (Android)
- [ ] Enabled ProGuard (Android)
- [ ] Set up log rotation
- [ ] Regular security updates
- [ ] Backup strategy in place

## Regular Security Tasks

**Daily:**
- Monitor API logs for unusual activity
- Check error logs

**Weekly:**
- Review new user/API registrations
- Update usage statistics

**Monthly:**
- Security updates
- Password rotation (recommended)
- Backup verification

**Quarterly:**
- Security audit
- Review access controls
- Update dependencies

---

**Security is an ongoing process, not a one-time task!**
