package com.change.freevpn.adapters;

import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.ImageView;
import android.widget.TextView;

import androidx.annotation.NonNull;
import androidx.cardview.widget.CardView;
import androidx.recyclerview.widget.RecyclerView;

import com.bumptech.glide.Glide;
import com.change.freevpn.R;
import com.change.freevpn.models.VpnServer;

import java.util.List;

public class ServerAdapter extends RecyclerView.Adapter<ServerAdapter.ServerViewHolder> {

    private List<VpnServer> servers;
    private OnServerClickListener listener;
    private VpnServer selectedServer;

    public interface OnServerClickListener {
        void onServerClick(VpnServer server);
    }

    public ServerAdapter(List<VpnServer> servers, OnServerClickListener listener) {
        this.servers = servers;
        this.listener = listener;
    }
    
    public void setSelectedServer(VpnServer server) {
        this.selectedServer = server;
    }

    @NonNull
    @Override
    public ServerViewHolder onCreateViewHolder(@NonNull ViewGroup parent, int viewType) {
        View view = LayoutInflater.from(parent.getContext())
                .inflate(R.layout.item_server, parent, false);
        return new ServerViewHolder(view);
    }

    @Override
    public void onBindViewHolder(@NonNull ServerViewHolder holder, int position) {
        VpnServer server = servers.get(position);
        
        holder.tvServerName.setText(server.getServerName());
        holder.tvCountry.setText(server.getCountry());
        
        // Load flag image
        if (server.getFlagUrl() != null && !server.getFlagUrl().isEmpty()) {
            Glide.with(holder.itemView.getContext())
                    .load(server.getFlagUrl())
                    .placeholder(R.drawable.ic_flag_placeholder)
                    .error(R.drawable.ic_flag_placeholder)
                    .into(holder.ivFlag);
        } else {
            holder.ivFlag.setImageResource(R.drawable.ic_flag_placeholder);
        }
        
        // Highlight if selected or connected
        int backgroundColor;
        if (server.isConnected()) {
            backgroundColor = holder.itemView.getContext().getColor(R.color.success_light);
        } else if (server.isSelected()) {
            backgroundColor = holder.itemView.getContext().getColor(R.color.accent_light);
        } else {
            backgroundColor = holder.itemView.getContext().getColor(R.color.white);
        }
        holder.cardView.setCardBackgroundColor(backgroundColor);
        
        // Highlight selected server
        if (selectedServer != null && selectedServer.getId() == server.getId()) {
            holder.itemView.setBackgroundColor(0xFFE8F5E9); // Light green
        } else {
            holder.itemView.setBackgroundColor(0xFFFFFFFF); // White
        }
        
        holder.itemView.setOnClickListener(v -> {
            if (listener != null) {
                listener.onServerClick(server);
            }
        });
    }

    @Override
    public int getItemCount() {
        return servers.size();
    }

    static class ServerViewHolder extends RecyclerView.ViewHolder {
        CardView cardView;
        ImageView ivFlag;
        TextView tvServerName;
        TextView tvCountry;

        ServerViewHolder(@NonNull View itemView) {
            super(itemView);
            cardView = (CardView) itemView;
            ivFlag = itemView.findViewById(R.id.iv_flag);
            tvServerName = itemView.findViewById(R.id.tv_server_name);
            tvCountry = itemView.findViewById(R.id.tv_country);
        }
    }
}
