package com.change.freevpn.vpn;

import android.app.Activity;
import android.content.Context;
import android.content.Intent;
import android.net.VpnService;
import android.util.Log;

/**
 * VPN Manager - Handles VPN connection lifecycle
 */
public class VpnManager {
    private static final String TAG = "VpnManager";
    private static final int VPN_REQUEST_CODE = 100;
    
    private Context context;
    private Activity activity;
    private String currentConfig;
    private boolean isConnected = false;
    
    public VpnManager(Context context, Activity activity) {
        this.context = context;
        this.activity = activity;
    }
    
    /**
     * Start VPN connection with OpenVPN config
     */
    public void connect(String ovpnConfig, VpnCallback callback) {
        this.currentConfig = ovpnConfig;
        
        // Check VPN permission
        Intent vpnIntent = VpnService.prepare(context);
        
        if (vpnIntent != null) {
            // Need to request permission
            if (callback != null) {
                callback.onPermissionRequired(vpnIntent, VPN_REQUEST_CODE);
            }
        } else {
            // Permission already granted, start VPN
            startVpnService(ovpnConfig);
            if (callback != null) {
                callback.onStarting();
            }
        }
    }
    
    /**
     * Handle permission result from activity
     */
    public void onActivityResult(int requestCode, int resultCode, VpnCallback callback) {
        if (requestCode == VPN_REQUEST_CODE) {
            if (resultCode == Activity.RESULT_OK) {
                // Permission granted, start VPN
                startVpnService(currentConfig);
                if (callback != null) {
                    callback.onStarting();
                }
            } else {
                // Permission denied
                if (callback != null) {
                    callback.onError("VPN permission denied");
                }
            }
        }
    }
    
    /**
     * Start the VPN service
     */
    private void startVpnService(String config) {
        try {
            Intent intent = new Intent(context, OpenVpnService.class);
            intent.putExtra("config", config);
            context.startService(intent);
            isConnected = true;
            Log.i(TAG, "VPN service started");
        } catch (Exception e) {
            Log.e(TAG, "Failed to start VPN service", e);
        }
    }
    
    /**
     * Stop VPN connection
     */
    public void disconnect() {
        try {
            Intent intent = new Intent(context, OpenVpnService.class);
            context.stopService(intent);
            isConnected = false;
            Log.i(TAG, "VPN service stopped");
        } catch (Exception e) {
            Log.e(TAG, "Failed to stop VPN service", e);
        }
    }
    
    /**
     * Check if VPN is connected
     */
    public boolean isConnected() {
        return isConnected;
    }
    
    /**
     * Set connection status
     */
    public void setConnected(boolean connected) {
        this.isConnected = connected;
    }
    
    /**
     * Get VPN request code for activity result
     */
    public static int getVpnRequestCode() {
        return VPN_REQUEST_CODE;
    }
    
    /**
     * Callback interface for VPN events
     */
    public interface VpnCallback {
        void onPermissionRequired(Intent intent, int requestCode);
        void onStarting();
        void onConnected();
        void onDisconnected();
        void onError(String error);
    }
}
